'use strict'
document.addEventListener("DOMContentLoaded", function () {

    const iawDefault = {
        merchant: 'WIDGET', // Пример мерчанта для ифт
        btnId: 'iawBtn',    // Пример идентификатора для кнопок
        customer: { 
            email: "customer@mail.ru"
        },
        bundle: { 
            fiscalInfo: { 
                clientEmail: "client_email@yandex.ru" 
            }, 
            items: [ 
                { 
                    positionId: 1, 
                    name: "Обеденный стол", 
                    code: "арт.4957686074846", 
                    description: "Стол барный Dorian 11770", 
                    shippable: true, 
                    price: 777.0, 
                    measure: 0, 
                    quantity: 1, 
                    mass: 2400, 
                    taxParams: { 
                    taxType: "vat20" 
                    }, 
                    paymentType: "full_prepayment", 
                    paymentSubject: 1, 
                    amount: 777.0, 
                    discount: 0, 
                    userData: "Дополнительный реквизит предмета расчета" 
                } 
            ] 
        } 
    };

    class vtbIAW {
        constructor(options) {
            this.iawClose = document.createElement('div');
            this.frameWrapper = document.createElement('div')
            this.iawCloseBtnWrapper = document.createElement('div');
            this.iawCloseBtn = document.createElement('button');
            this.iawFrame = document.createElement('iframe');
            this.iawStyle = document.createElement('link');
            this.iawBtn = options.btnId;
            this.iawUrl = 'https://if-pay-test.vtb.ru/api/tsp-ia-box-cloud-cms/v1/orders';
            this.iawMerchant = options.merchant;
            this.iawCustomer = options.customer;
            this.iawBundle = options.bundle;

            if (this.iawMerchant.length <= 0) {
                throw new Error('[IAW] Credential: Merchant lost');
            }

            if (this.iawBtn.length <= 0) {
                throw new Error('[IAW] Button ID cannot be empty');
            }

            this.#init();
        }

        #getAttr(target) {
            return { price: target.attributes['data-price'].value, id: this.#getUid() }
        }

        #generateRandomCode() {
            let code = "";
            for (let i = 0; i < 4; i++) {
                const randomDigit = Math.floor(Math.random() * 10); // от 0 до 9
                code += randomDigit;
            }
            return code;
        }

        #getUid() {
            const date = new Date();
            const year = date.getFullYear();
            const month = String(date.getMonth() + 1).padStart(2, '0');
            const day = String(date.getDate()).padStart(2, '0');
            const hour = String(date.getHours()).padStart(2, '0');
            const minute = String(date.getMinutes()).padStart(2, '0');
            const second = String(date.getSeconds()).padStart(2, '0');
            const millisecond = String(date.getMilliseconds()).padStart(3, '0');

            return `WT${this.#generateRandomCode()}${year}${month}${day}${hour}${minute}${second}${millisecond}${this.#generateRandomCode()}`;
        }

        #setStyle() {
            this.iawStyle.setAttribute("href", "./public/css/iaw.css");
            this.iawStyle.setAttribute("rel", "stylesheet");
            document.head.appendChild(this.iawStyle);
        }

        #setFrame(url) {
            try {
                if (url.length > 0) {
                    let attrs = {
                        src: url,
                        class: 'iawFrame',
                        allowpaymentrequest: 'true',
                        allowtransparency: 'true',
                        frameborder: '0',
                        loading: 'lazy'
                    };

                    Object.entries(attrs).forEach(([attr, value]) =>
                        this.iawFrame.setAttribute(attr, value),
                    );
                    this.frameWrapper.className = 'frameWrapper'
                    this.#setClose()
                    this.frameWrapper.appendChild(this.iawFrame)
                    this.iawClose.appendChild(this.frameWrapper);
                    this.iawClose.setAttribute('class', 'iawShow');
                }
            } catch (error) {
                throw new Error(`[IAW] Frame link cannot be null or Frame cannot be load target page: ${error}`);
            }
        }

        #setClose() {
            this.iawCloseBtn.setAttribute('type', 'button');
            this.iawCloseBtn.setAttribute('class', 'iawCloseButton');
            this.iawCloseBtnWrapper.setAttribute('class', 'iawCloseBtnWrapper');
            this.iawCloseBtnWrapper.appendChild(this.iawCloseBtn);
            this.frameWrapper.appendChild(this.iawCloseBtnWrapper);
            document.body.appendChild(this.iawClose);
        }

        #setListener() {
            let button = document.querySelector(`#${this.iawBtn}`);
            let _this = this;

            _this.iawCloseBtnWrapper.addEventListener('click', function (e) {
                e.preventDefault();
                e.stopPropagation();
                _this.iawClose.remove();
            });

            button.addEventListener('click', async function (e) {
                e.preventDefault();
                e.stopPropagation();

                let payData;

                payData = _this.#getAttr(e.target);
                await _this.#sendAction(payData);
            });
        }

        async #sendAction(options) {
            let _this = this, timestamp = new Date(), args, price;

            if (options.price) {
                price = options.price.replace(',','.');
            }

            timestamp.setDate(timestamp.getDate() + 1);
            timestamp = timestamp.toISOString();

            args = {
                'orderId': options.id,
                'orderName': `№ ${options.id}`,
                'expire': timestamp,
                'amount': {
                    'value': parseFloat(price),
                    'code': 'RUB'
                }
            }

            _this.#sendRequest(
                'post',
                _this.iawUrl,
                args
            ).then(data => {
                try {
                    if (data) {
                        _this.#setFrame(data.object.payUrl);
                    }
                } catch (error) {
                    throw new Error(`[IAW] Order do not created: ${error}`);
                }
            }).catch(error => {
                throw new Error(`[IAW] Request failed: ${error}`);
            });
        }

        async #sendRequest(method, url, body = null) {
            let data = JSON.stringify(body);
            let _this = this,
                result,
                options = {
                    method: method.toUpperCase(),
                    headers: {
                        'Content-Type': 'application/json',
                        'Content-Length': data.length ? data.length : 0,
                        // 'Merchant-Host': location.origin, // для прода
                        'Merchant-Host': 'https://widget2.ecom.tspvtb.ru', // хост для ифт
                        'Merchant-Authorization': _this.iawMerchant,
                    },
                    redirect: 'follow',
                };

            if (body !== null) {
                options.body = data;
            }

            result = await fetch(url, options);
            return result.json();
        }

        #init() {
            this.#setStyle();
            this.#setListener();
        }
    }

    new vtbIAW((typeof iawOptions !== 'undefined') ? iawOptions : iawDefault);
});
